function  U = get_U_newton(U,D,dx,dt,tol,dirichlet_value)
% this function performs a newton algorithm to find the next iterate of a fully implicit scheme of the burger equation.


U0 = U;        % keeping value from the previous time step

Vk = U;        % initialisation of Vk 

norm_ = 2*tol; % initialisation (we have to enter the loop)

it = 0;        % initial number of iteration

it_max = 20 ;  % maximal number of iteration

while norm_ > tol && it < it_max

Jac = create_jacobian(Vk,D,dx,dt);
F   = create_F(Vk,U0,D,dx,dt,dirichlet_value);

% Solving the linear problem for the newton iteration.
deltaV = - Jac\F ;
% finding the new iterate 
Vk_new   = Vk + deltaV ;

% hard coding the dirichlet value, otherwise if the initial data does not
% fit with the dirichlet value, V(1) will never get to the dirichlet value
Vk_new(1)= dirichlet_value;
% relative error.
norm_ = norm(Vk_new-Vk)/norm(Vk) ;

Vk = Vk_new;
it = it+1;
end

U = Vk;

pprintf('after %d (newton) iteration, norm = %e \n',it,norm_ )

end



function jac = create_jacobian(U,D,dx,dt)
% this function create the jacobian matrix used by a Newton algorithm to solve the fully implicit discretization of the Burger equation.
% the boundary at x = 0 is a Dirichlet boundary for the Burger problem -> the linear problem solved at each Newton iteration has then a homogeneous Dirichlet BC at x = 0.
% the boundary at x = 1 is a Neumann   boundary for the Burger problem -> the linear problem solved at each Newton iteration has then a homogeneous Neumann   BC at x = 1.

% initialisation of the sparse matrix.
jac = sparse(length(U),length(U));


%diffusion coefficient
diff_coeff = D*dt/dx/dx;
%convection coefficient
conv_coeff = dt/2/dx;

%
%  FILLING THE MATRIX
%


% filling the matrix with partial derivative coefficients.
for i= 2:length(U)-1
 jac(i,i-1) =   -   diff_coeff  - conv_coeff *  U(i);
 jac(i,i)   = 1 + 2*diff_coeff  + conv_coeff * (U(i+1)-U(i-1));
 jac(i,i+1) =   -   diff_coeff  + conv_coeff *  U(i);
end


% dirichlet BC
jac(1,1)=1;

% Applying the Neumann BC, centered Scheme (V_(N+1) - V_(N-1))/2/dx =0 using the ghost point. 
% The commented part is in the case the uncentered scheme is applied i.e. (V_(N+1) - V_(N))/dx =0
jac(end,end-1) = -   2*diff_coeff  ;%   - diff_coeff - conv_coeff *  U(end);
jac(end,end)   = 1 + 2*diff_coeff  ;% 1 + *diff_coeff + conv_coeff * (2*U(end)-U(end-1));




end


function F = create_F(Vk,U0,D,dx,dt,dirichlet_value)
% this function create an evaluation of the function F needed by the Newton algorithm to solve the fully implicit discretisation of the Burger equation.
% the boundary at x = 0 is a Dirichlet boundary for the Burger problem -> the linear problem solved at each Newton iteration has then a homogeneous Dirichlet BC at x = 0.
% the boundary at x = 1 is a Neumann   boundary for the Burger problem -> the linear problem solved at each Newton iteration has then a homogeneous Neumann   BC at x = 1.


V_i_minus_one = [dirichlet_value ;Vk(1:end-1)]; %here we keep the dirichlet_value of the Burger problem because it is needed in building F (for F(2) in particular).

% Applying the Neumann BC, centered Scheme (V_(N+1) - V_(N-1))/2/dx =0 using the ghost point. 
% The commented part is in the case the uncentered scheme is applied i.e. (V_(N+1) - V_(N))/dx =0
V_i_plus_one = [ Vk(2:end); Vk(end-1) ]; %[ Vk(2:end); Vk(end) ];


F = Vk-U0 -dt*D/dx/dx*( V_i_minus_one - 2*Vk + V_i_plus_one) ...
    +      dt/2/dx*Vk.*( V_i_plus_one-V_i_minus_one );

% homogeneous dirichlet BC for the Newton iteration.
F(1) = 0;
end








































































































































% AUTHOR : Pierre-Emmanuel des Boscs
% Date   : June 16th 2017
% Code for exercise 7.