%%          SETTINGS (TO BE PLAYED WITH)
% ----------------------------------------------------------------------- %
% Geometry
L_y= 25;                % Length of the domain in y (km)
L_x= 25;                % Length of the domain in x (km)
L_i= 3;                 % Length of initially infected area (km)
L_s= 8;                 % Length of safety zone (km)

% Discretization
I  = 030;               % Number of grid points in y   
J  = 030;               % Number of grid points in y and x

% Initial condition
H_0= 1.888776e6;        % Initial number of humans
Z_0= 1e4;               % Initial number of zombies

% Diffusion
diffusion = true;       % Turns diffusion on(true)/off(false)
D_H= 1e-1;              % Diffusivity of humans (km2/h)
D_Z= 3e-1;              % Diffusivity of zombies(km2/h)               

% Convection
convection = true;      % Turns convection on(true)/off(false)
vmax= 0.15;             % Maximum migration velocity of humans (km/h)

% Reaction
reaction = true;        % Turns reaction on(true)/off(false)
a  = 4e-6;              % Deadliness of humans 
b  = 5e-6;              % Deadliness of zombies
c  = 6e-6;              % Infectiousness of zombies
as = 1e-2;              % Deadliness of humans in safety zone
bs = 1e-8;              % Deadliness of zombies in safety zone
cs = 1e-8;              % Infectiousness of zombies in safety zone

% Iterative solver 
it_max= 1e5;        % Maximum number of iterations of Newton algorithm
tol_N = 1e-5;       % Tolerance of Newton algorithm

% Visualization
nshow = 5;          % Plotting period (increase for speed-up)
Hmax = 2;           % Scale of human concentration graph (zlim=Hmax*h0)
Zmax = 0.1;         % Scale of zombie concentration graph (zlim=Zmax*z0)


% ----------------------------------------------------------------------- %
%%          INITIALIZATION (ONLY FOR INTERESTED ONES)
% ----------------------------------------------------------------------- %
% Discretization
N = I * J;                              %Total number of degrees of freedom
dy= L_y/(I-1);                          % Grid spacing in y
dx= L_x/(J-1);                          % Grid spacing in x
[x,y]=meshgrid(0:dx:L_x,L_y:-dy:0);     % coordinates of each grid node
left   = find(x(:)==0  );               % Indices of boundary nodes
right  = find(x(:)==L_x);    
bottom = find(y(:)==0  );    
top    = find(y(:)==L_y); 
SZ = x>=L_x-L_s & y>=L_y-L_s;           % Indices of safety zone
Id = speye(N);                          % Identity matrix

% Initial condition
h0 = H_0 / (L_x*L_y);                   % Initial human density
z0 = Z_0 / L_i^2;                       % Initial zombie density
H = h0 .* ones(I,J);                    % Initial condition
Z = z0 .* (x<=2) .* (y<=2);
U = [ H(:) ; Z(:) ];                    % Global solution vector

% Convection
v_x=x/L_x.*(1-x/L_x)*sqrt(8)*vmax*convection; % Migration field of humans 
v_y=y/L_y.*(1-y/L_y)*sqrt(8)*vmax*convection; % (km/h)
div=2*(L_x^2*L_y-2*y*L_x^2+L_x*L_y^2-2*x*L_y^2)/...
      (5*L_x^2*L_y^2);                  % Divergence of migration field

% Reaction
alpha = (b+c)*ones(I,J)*reaction ;      % Removal rate of humans
beta  = (c-a)*ones(I,J)*reaction ;      % Creation rate of zombies
alpha(SZ) = (bs+cs)*reaction;           % Removal rate of H in safety zone
beta(SZ) = (cs-as)*reaction;            % Creation rate of Z in safety zone
dt = min(0.1/z0/max(alpha(:))/(dx*dy),1);% Step size for reaction

% Iterative solver initialization
n = 0;                                  % Index of time step
delta_U = Inf;                          % Steady state measure


% ----------------------------------------------------------------------- %
%%           DISCRETE DERIVATIVES
% ----------------------------------------------------------------------- %
[D_x1,D_y1,D_x2,D_y2] = FDMatrices(I,N,dx,dy,left,right,bottom,top);


% ----------------------------------------------------------------------- %
%%           COMPUTATION
% ----------------------------------------------------------------------- %
while 1
    
    % Show solution
    if mod(n,nshow)==0
        ShowDistribution(U,Hmax*h0,Zmax*z0,n*dt,I,J,N,x,y,L_x,L_y);  
    end
    
    % Compute next time-step
    V = Newton(U,tol_N,it_max,D_x1,D_y1,D_x2,D_y2,Id,D_H,D_Z,v_x,v_y,...
               alpha,beta,dt,div,N);

    % Proceed to next step
    U = V;
    n = n+1;
    
end

fprintf('Steady state reached \n');