%% Code hygiena - removes remainders of previous calculations
clear;
close all;

%% Settings

% Spatial discretisation
N = 1001;                   			% Number of spatial computational points
x_min = 0;                  			% Position of left boundary [m]
x_max = 10000;              			% Position of right boundary [m]
dx = (x_max-x_min)/(N-1);   			% Step size [m]
x = [x_min : dx : x_max ]';             % Main spatial grid
x_ = [x_min+dx/2 : dx : x_max-dx/2]';   % Staggered spatial grid


% Temporal discretisation
t_end_hours = 100;          			% Time of the end of simulation [h] 
CFL = 0.3;                  			% Courant number


% Boundary conditions
h_left  = @(t) 10;          			% water depth at x=x_min [m]
% h_left  = @(t) 10 + 0.5*sin(2*pi*t/(120)); 
                            			% uncomment the line above to introduce 
                            			% fluctuations of water depth on left boundary


% Specification of the channel
w_left  = 3;                			% Width at x=x_min [m]
w_right = 3;                			% Width at x=x_max [m]
w = (linspace(w_left,w_right,N))';  	% Creates a linear variation of the 
                                    	% channel width from w_left to w_right
b_left = 140;               			% Altitude of channel bed at x=x_min [m]
b_right = 100;              			% Altitude of the channel bed at x=x_max [m]
b = (linspace(b_left,b_right,N))';  	% Channel bed profile
b = b + 0.1 *exp(-(x-(x_max+x_min)/2).^2 *(0.0000001) ).* cos(2*pi*x/(x_max/10));   
                            			% Adds small oscilations to the channel bed profile


% Frictional losses
n = 0.01 ;                  			% Manning's friction coefficient


%% Initialization

% Water depth
h_initial = 0.5;
h = ones(N,1)*h_initial + (h_left(0)-h_initial)*(x<0.25*x_max); 
                            					 % Initial water depth profile (step function)

% Velocity
S_b = mean( (b(2:end)-b(1:end-1)) ./ (dx) ) ;    % Channel bed slope [m/m]
R = mean( h.*w / (w + 2.*h) );                   % Hydraulic radius [m]
u_initial = sqrt( abs(S_b) * R(1)^(4/3) / n^2 ); % Calculates the steady state velocity for 
													%given bed slope [m/s]
u = u_initial .* ones(N-1,1);                    % Initial velocity vector [m/s]

% Initial axes limits for the graph of velocity
u_min_plot = min(u)-1 ;
u_max_plot  = max(u) +1;

% Time
t = 0;                                          % Initial time
t_end = 3600 * t_end_hours;                     % End time in seconds

% Iteration counter
m = 1;                                          % Counts the time steps


%% Computation
while t < t_end
    % Calculation of time-step to satisfy CFL condition
    dt = CFL * dx / max(max(u),1);
    
    
    % Calculation of water depth
    [h] = solve_continuity(h,u,dx,dt,h_left(t));
        
    
    % Updating velocity (solving momentum balance)
    [u] = solve_momentum_transport(u,h,w,n,dx,dt,b);

    
    % Advancing in time
    m=m+1; t=t+dt;

    
    % Visualization
    if(mod(m,100)==0)   						% modify to change the plotting frequency
    figure(1);
    u_min_plot = max(0.8*u_min_plot,min(u)-1); 	% The extent of the vertical 
    u_max_plot = min(1.2*u_max_plot,max(u)+1);   %axis of the graph of velocity
    subplot(2,1,1); 
        area(x,h+b,'FaceColor',[175,238,238]./256); 
        hold on; 
        area(x,b,'FaceColor',[222,184,135]./256); 
        axis([x_min x_max b_right (b_left+h_left(0))]); 
        legend('water','ground'); 
        title(['t = ' num2str(t/60) ' min']); 
        hold off;
        xlabel('Horizontal position [m]');
        ylabel('Altitude [m]');
    subplot(2,1,2); 
        plot(x_,u); 
        legend('flow velocity'); 
        axis([x_min x_max u_min_plot, u_max_plot]);
        xlabel('Horizontal position [m]');
        ylabel('Velocity [m/s]');
    drawnow;
    pause(0.01);
    end
end
