%% Code hygiena
clear;      %removes current variables from workspace
close all;  %closes all open figures


%% Settings

% Spatial discretisation
N = 1000;                   % number of spatial computational points
x_min = 0;                  % position of left boundary [m]
x_max = 1000;               % position of right boundary [m]


% Temporal discretisation
t_end_hours = 100;          % Time of the end of simulation [h] 
CFL = 0.5;                  % Courant number


% Boundary conditions
h_left  = 0;                % water depth at x = x_min [m] 


% Initial condition
h = @(x) ones('like',x) * h_left + 0.1 .* gaussmf(x, [0.04*x_max 0.5*x_max]); 
                            % initial water depth at t = 0

% Convection velocity
u = 10* ones(N-1,1); % vector of velocity u(x), stored in a staggered grid
                      

% Plotting frequency
f = 10;                     % controls the speed of the visual output


%% Initialization

% Spatial grid
dx = (x_max-x_min)/(N-1);   % calculation of grid spacing
x = [x_min : dx : x_max ]'; % generation of spatial grid


% Numerical solution
h_num = h(x);


% Time
t_end = 3600 * t_end_hours; % end time in seconds
t = 0;                      % initial time


% Counter of time steps
m = 1;


% Axes limits for visual output
h_max = max(h_num);
h_min = min(h_num);


%% Computation

while t < t_end
    
    % Calculation of time-step to satisfy CFL condition
    dt = CFL * dx / max(max(u),1);
    
    
    % Updating water depth (numerically)
    [h_num] = solve_continuity(h_num,u,dx,dt,h_left);
    
    
    % Analytical solution
    h_a = h(x - u(1).*(t+dt));
    
    
    % Advancing in time
    m=m+1; 
    t=t+dt;
    
    
    % Visualization
    if(mod(m,f)==0)
        figure(1);
        plot(x,h_num,x,h_a); 
        title(['t = ' num2str(t/60) ' min']); 
        legend('numerical','analytical');
        axis([x_min x_max h_min h_max]);
        drawnow;    
        pause(0.1)
    end
    

end
